-------------------------------------------------
-- 🌐 Betriebsmittelausgabe WEB-Modul (Schema: API)
-------------------------------------------------

-- HTML-Frontend per Funktion als Text und per Postgrest-API verfügbar machen
-- Diese Seite ist die Hauptseite für die Betriebsmittelausgabe, die eine Übersicht über alle Betriebsmittel bietet, die ausgegeben werden können.
-- Sie enthält Filter-Selektoren für den Status und Typ der Betriebsmittel, eine Haupttabelle zur Anzeige der Betriebsmittel und eine Floating Action Bar für die Statusänderung.
CREATE OR REPLACE FUNCTION api.artinventarausgabe_page()
  RETURNS "text/html"
  AS $BODY$
  SELECT concat_ws( E'\n',
  '<!DOCTYPE html>', '<html lang="de">',
  api.html_head_get('Betriebsmittelausgabe'),
  $html$
  <body class="bg-gray-50 text-gray-800">

    <!-- Navigation -->
    <nav id="mainNavbar" class="fixed top-0 left-0 w-full z-50 bg-white shadow px-6 py-2 sm:py-4 flex justify-between items-center">
      <div>
        <div class="flex items-center gap-2">
          <svg class="w-7 h-7 text-blue-600" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
            <rect width="20" height="14" x="2" y="7" rx="2" />
            <path d="M16 3v4M8 3v4m-6 4h20" />
          </svg>
          <h1 class="text-xl font-semibold tracking-tight">Betriebsmittelausgabe</h1>
        </div>
        <span class="ml-9 text-xs text-gray-400 font-semibold tracking-wide">Prodat ERP</span>
      </div>
      <div class="flex items-center gap-4">
        <!-- User Dropdown Trigger -->
        <div id="userDropdownWrap" class="relative hidden">
          <button id="userMenuBtn" type="button" class="flex items-center gap-2 px-2 py-1 rounded-xl hover:bg-gray-100 focus:outline-none transition">
            <!-- Avatar Initialen -->
            <div id="userAvatar" class="w-9 h-9 flex items-center justify-center bg-gradient-to-tr from-green-400 to-green-700 text-white text-lg font-bold rounded-full shadow-sm"></div>
            <!-- Username & Badge -->
            <div class="flex flex-col items-start">
              <span id="userName" class="text-base font-semibold text-gray-800"></span>
              <span id="adminBadge" class="flex items-center gap-1 text-xs font-medium bg-green-100 text-green-700 px-2 py-0.5 rounded hidden">
                <svg class="w-4 h-4 text-green-600" fill="none" stroke="currentColor" stroke-width="3" viewBox="0 0 24 24">
                  <title>Admin-Berechtigung</title>
                  <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7"/>
                </svg>
                Admin
              </span>
            </div>
            <svg class="w-4 h-4 ml-1 text-gray-400 transition" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" d="M19 9l-7 7-7-7"></path>
            </svg>
          </button>
          <!-- Dropdown-Menu -->
          <div id="userDropdown" class="z-30 absolute right-0 mt-2 w-44 bg-white rounded-xl shadow-lg border border-gray-100 opacity-0 pointer-events-none scale-95 transition-all duration-200">
            <div class="py-2">
              <button id="logoutDropdownBtn" class="w-full flex items-center px-4 py-2 text-sm text-red-600 hover:bg-red-50 rounded-b-xl transition">
                <svg class="w-4 h-4 mr-2 text-red-400" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
                  <path stroke-linecap="round" stroke-linejoin="round" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1m0-9V7" />
                </svg>
                Abmelden
              </button>
            </div>
          </div>
        </div>
        <button id="loginBtn"  class="hidden px-4 py-2 border border-gray-400 rounded hover:bg-gray-100        ">Login</button>
      </div>
    </nav>

    <main id="pageContent" class="flex flex-col max-w-none mx-auto px-4 py-6">

      <!-- Filter-Selektoren -->
      <section class="bg-white rounded-2xl shadow-xl border border-blue-100 p-6">
        <div class="flex flex-col md:flex-row md:gap-6">
          <div class="w-auto md:overflow-y-auto md:h-[calc(100vh-70px-110px)] md:w-64 flex-shrink-0">
            <label class="block mb-2 text-base font-medium text-blue-700">Status-Filter für Tabelle</label>
            <div id="selektor_status"></div>
            <div id="selektor_typ"></div>
          </div>
        <!-- Haupttabelle -->
        <div id="fa-container" class="flex-1 min-w-0 p-0"></div>
      </section>
    </main>

    <!-- Floating Action Bar -->
    <div id="floating-action-bar"
         class="fixed right-4 bottom-6 flex items-center gap-3 bg-blue-700 text-white px-6 py-3 rounded-2xl shadow-xl transition-all duration-300 opacity-0 pointer-events-none z-50"
         style="min-width:220px">
      <span id="selected-counter" class="font-semibold text-lg"></span>
      <button id="clear-selection-btn"
              class="flex items-center gap-2 px-4 py-2 rounded bg-red-50 text-red-700 font-semibold hover:bg-red-100 shadow"
              style="min-width: 140px;">
        <svg class="w-5 h-5 text-red-500" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
        </svg>
        Auswahl löschen
      </button>
      <button id="change-status-btn"
              class="flex items-center gap-2 px-4 py-2 rounded bg-white text-blue-700 font-semibold hover:bg-blue-100 shadow"
              style="min-width: 140px;">
        <svg class="w-5 h-5 text-blue-500" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7" />
        </svg>
        Status setzen
      </button>
    </div>

    <dialog id="status-dialog" class="overflow-visible w-128 rounded-xl shadow-xl bg-white p-6">
      <form method="dialog">
        <h3 class="text-lg font-semibold mb-4">Neuen Status wählen</h3>
        <select id="new-status" name="new-status" class="w-full border rounded px-2 py-2 mb-4">
          <option value="entnahme">Entnahme</option>
          <option value="rueckgabe">Rückgabe</option>
          <option value="zur kalibrierung">Zur Kalibrierung</option>
          <option value="von kalibrierung">Von Kalibrierung</option>
          <!-- beliebig erweiterbar -->
        </select>

        <!-- Zusätzliche Felder -->
        <div id="entnahme-fields" class="mb-4">
          <input
            type="text"
            id="pr_ort"
            prodat="pr_ort"
            qrcode=true
            name="pr_ort"
            class="w-full border rounded px-2 py-2 mb-2"
            placeholder="Kostenstelle"
          />
          <input
            type="integer"
            id="pr_abix"
            prodat="pr_abix"
            qrcode=true
            name="pr_abix"
            class="w-full border rounded px-2 py-2"
            placeholder="ABK"
          />
        </div>

        <div class="flex justify-end gap-2">
          <button type="button"
                  onclick="document.getElementById('status-dialog').close()"
                  class="flex items-center gap-2 px-4 py-2 rounded bg-red-50 text-red-700 font-semibold hover:bg-red-100 shadow">
            <svg class="w-5 h-5 text-red-500" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" d="M6 18L18 6M6 6l12 12" />
            </svg>
            Abbrechen
          </button>

          <button type="submit"
                  class="flex items-center gap-2 px-4 py-2 rounded bg-blue-600 text-white font-semibold hover:bg-blue-700 shadow"
                  name="submit_action" value="save">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" d="M5 13l4 4L19 7" />
            </svg>
            Speichern
          </button>

          <button type="submit"
                  class="flex items-center gap-2 px-4 py-2 rounded bg-blue-700 text-white font-semibold hover:bg-blue-800 shadow"
                  name="submit_action" value="save_logout">
            <svg class="w-5 h-5 text-white" fill="none" stroke="currentColor" stroke-width="2" viewBox="0 0 24 24">
              <path stroke-linecap="round" stroke-linejoin="round" d="M17 16l4-4m0 0l-4-4m4 4H7m6 4v1m0-9V7" />
            </svg>
            Speichern & Ausloggen
          </button>
        </div>
      </form>

    </dialog>

    <!-- Toast-Benachrichtigungen -->
    <div id="toast-container" class="fixed bottom-4 right-4 flex flex-col gap-2 z-50 w-[384px] max-w-full"></div>

    <script type="module">
      import { renderTable }    from './file?path=component_table_clusterize.js';
      import { renderCards }    from './file?path=component_cards.js';
      import { renderSelector } from './file?path=component_selector.js';

      // Navbarhöhe setzen
      function setNavbarSpacer() {
          const navbar = document.getElementById('mainNavbar');
          const pageContent = document.getElementById('pageContent');
          const table_container = document.getElementById('fa-container');
          if (!navbar || !pageContent) return;
          const height = navbar.offsetHeight;
          pageContent.style.marginTop = height + 'px';
          pageContent.classList.add('h-[calc(100vh-' + height + 'px)]');
          // table_container.classList.add('overflow-x-auto', 'md:h-[calc(100vh-' + height + 'px-204px)]', 'h-[calc(100vh-' + height + 'px-48px)]')
        }

      document.addEventListener('DOMContentLoaded', function() {
        setNavbarSpacer();
        window.addEventListener('resize', setNavbarSpacer);
      });

      // Auswahl rendern

      // Hilfsfunktion für API-Call (mit Auth wenn nötig)
      async function fetchAuswahl() {
        const response = await fetchWithAuth('./execute_standard_sql_to_jsonb_set', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ standard_sql: 'Sondermodul.ArtInventar_Ausgabe_Standardfilter' })
        });
        if (!response.ok) throw new Error('Fehler beim Laden der Filterauswahl! Angemeldet?');
        // Hier je nach API-Antwort entweder .json() oder .json().result etc.
        const arr = await response.json();
        // arr ist ein Array mit einem Element, das 'auswahl' enthält!
        const auswahl = arr[0]?.auswahl;
        if (!auswahl) throw new Error('Falsches API-Format: "auswahl" fehlt!');
        return auswahl;
      }

      //// Status definieren
      let selectedStatus = [];
      let selectedTyp = [];
      let preselectedSet = new Set();

      async function initSelectors() {
        try {
          const auswahl = await fetchAuswahl();
          // Debug
          console.log('[DEBUG] Auswahl vom Backend:', auswahl);

          renderSelector({
            containerId: 'selektor_status',
            items: auswahl.status,
            title: 'Status wählen (Mehrfachauswahl möglich)',
            onChange: selected => {
              selectedStatus = selected;
              reloadTable();
            }
          });

          renderSelector({
            containerId: 'selektor_typ',
            items: auswahl.typ,
            title: 'Typ wählen (Mehrfachauswahl möglich)',
            onChange: selected => {
              selectedTyp = selected;
              reloadTable();
            }
          });

        } catch (e) {
          alert('Fehler beim Laden der Auswahlmöglichkeiten! Angemeldet?');
          console.error(e);
        }
      }

      initSelectors();

      //
      function updateFloatingBar() {
        const bar = document.getElementById('floating-action-bar');
        const counter = document.getElementById('selected-counter');
        const count = preselectedSet.size;

        if (count > 0) {
          bar.style.opacity = 1;
          bar.style.pointerEvents = 'auto';
          counter.textContent = `${count} Werkzeug${count > 1 ? 'e' : ''} ausgewählt`;
          document.getElementById('change-status-btn').disabled = false;
        } else {
          bar.style.opacity = 0;
          bar.style.pointerEvents = 'none';
          counter.textContent = '';
          document.getElementById('change-status-btn').disabled = true;
        }
      }

      // Tabelle Inventar/Werkzeugübersicht rendern
      function reloadTable() {
        const filter = {
          status: selectedStatus,
          ak_ac: selectedTyp
        };
        console.log('Aktuelle Filter:', filter);
        renderTable({
          endpoint: './execute_standard_sql_to_jsonb_set',
          containerId: 'fa-container',
          title: 'Betriebsmittel',
          columns: [
            { field: '__select__', header: 'Auswahl', width: '40px', ident: 'ainv_id' },
            { field: 'ainv_aknr', header: 'Artikelnummer' },
            { field: 'ak_bez', header: 'Bezeichnung' },
            { field: 'ainv_nr', header: 'Inventarnummer' },
            { field: 'ainv_sernummer', header: 'Seriennummer' },
            { field: 'ainv_hest', header: 'Hersteller' },
            { field: 'ainv_mebe', header: 'Messbereich' },
            { field: 'ainv_ort', header: 'Ausgabeort' },
            { field: 'ainv_abix', header: 'Ausgabe-ABK' },
            { field: 'ainv_auda', header: 'Ausgabedatum', type: 'date' },
            { field: 'ainv_slort', header: 'Inventarlagerort' },
            { field: 'ainv_befu', header: 'Befund' },
            { field: 'abks', header: 'Messwerterfassung (ABK)' },
            //{ field: 'ainv_lagernd', header: 'lagernd', type: 'boolean' },
            { field: 'status', header: 'Status' },
            { field: 'typ', header: 'Typ' },
          ],
          sqlParameter: 'Sondermodul.ArtInventar_Ausgabe',
          additionalFilter: filter,
          refreshInterval: 0,
          preselectedSet: preselectedSet
        });
        setTimeout(updateFloatingBar, 200);
        setNavbarSpacer();
      }
      reloadTable();


      // Login Button Handler
      document.getElementById("loginBtn").addEventListener("click", function () {
        window.location.href = './login_page?redirect=artinventarausgabe_page';
      });

      // Row-Checkbox Handler um Ausgabe-Button zu (de-)aktivieren
      document.addEventListener('change', function (e) {
        if (e.target.classList.contains('row-checkbox')) {
          updateFloatingBar();
        }
      });

      function showEntnahmeFelder() {
        const isEntnahme = document.getElementById('new-status').value === 'entnahme';
        const fields = document.getElementById('entnahme-fields');
        const kostenstelle = document.getElementById('pr_ort');
        const abk = document.getElementById('pr_abix');
        if (isEntnahme) {
          fields.classList.remove('hidden');
          kostenstelle.required = true;
        } else {
          fields.classList.add('hidden');
          kostenstelle.required = false;
          kostenstelle.value = '';
          abk.value = '';
        }
      }

      // Ausgabe Button Handler um Dialog-Form anzuzeigen
      document.getElementById('change-status-btn').addEventListener('click', function () {
        showEntnahmeFelder();
        document.getElementById('status-dialog').showModal();
      });

      document.getElementById('fa-container-refresh').addEventListener('click', function () {
        setTimeout(updateFloatingBar, 200);
      });

      // Dialog zusätzliche Felder bei Entnahme einblenden
      document.getElementById('new-status').addEventListener('change', function() {
        showEntnahmeFelder();
      });

      // Dialog-Submit an Prodat schicken
      document.querySelector('#status-dialog form').addEventListener('submit', async function (e) {
        e.preventDefault();

        // FormData vom Formular holen
        const form = e.target;
        const formData = new FormData(form, e.submitter);

        // Alle ausgewählten IDs einsammeln:
        const selectedIds = Array.from(preselectedSet);

        // Debug
        console.log('[DEBUG] Formular-Daten:', formData);

        // Status holen (steht im Select im Formular)
        const newStatus = formData.get('new-status');
        formData.delete('new-status');

        // Alle restlichen Felder als params-Objekt:
        const params = {};
        for (const [key, value] of formData.entries()) {
          // Optional: Nur befüllen, wenn Wert nicht leer ist
          if (value !== '') {
            params[key] = value;
          }
        }

        // Backend-Aufruf, per fetch inkl. Auth-Header:
        const response = await fetchWithAuth('./artinventar_ausgabe__status_set', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            artinventar_ids: selectedIds,
            status: newStatus,
            params: Object.keys(params).length ? params : null
          })
        });
        const resp = await response.json();
        if (!response.ok) {
          showToast(`Fehler beim speichern! ${resp.message}`, "error");
          throw new Error(`Fehler beim speichern! ${resp.message}`);
        } else {
          showToast(`${resp.result}`, "success");
        }

        // Herausfinden, welcher Submit-Button geklickt wurde:
        const submitAction = formData.get('submit_action'); // "save" oder "save_logout"

        if (submitAction === 'save') {
          // Dialog schließen, Tabelle neu laden:
          document.getElementById('status-dialog').close();
          preselectedSet.clear();
          document.getElementById('fa-container-refresh').click();
        } else {
          // Cookie ungültig setzen und zu Login-Seite umleiten
          document.cookie = "jwt=; path=/; expires=Thu, 01 Jan 1970 00:00:00 GMT";
          window.location.href = './login_page?redirect=artinventarausgabe_page';
        }
      });

      async function applyUserRights() {
        const response = await fetchWithAuth('./execute_standard_sql_to_jsonb_set', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({ standard_sql: 'Sondermodul.ArtInventar_Ausgabe_Berechtigung' })
        });
        const data = await response.json();
        const user = data[0];
        const userDropdownWrap = document.getElementById("userDropdownWrap");
        const userNameSpan = document.getElementById("userName");
        const userAvatar = document.getElementById("userAvatar");
        const adminBadge = document.getElementById("adminBadge");
        const loginBtn = document.getElementById("loginBtn");

        if (!user) {
          // Nicht eingeloggt
          userDropdownWrap.classList.add("hidden");
          loginBtn.classList.remove("hidden");
          return;
        }

        // Eingeloggt
        userDropdownWrap.classList.remove("hidden");
        loginBtn.classList.add("hidden");

        // Initialen
        const initials = (user.ad_vorn?.charAt(0) || "") + (user.ad_name?.charAt(0) || "");
        userAvatar.textContent = initials || "U";
        // Name
        userNameSpan.textContent = user.ad_vorn + " " + user.ad_name;

        // Admin
        // const isAdmin = user.rolnames.includes("~SYS.Administratoren");
        const isAdmin = user.is_admin;
        if (isAdmin) {
          adminBadge.classList.remove("hidden");
        } else {
          adminBadge.classList.add("hidden");
        }
        // Navbar-Höhe setzen
        setNavbarSpacer();

        // Select-Optionen anpassen
        document.querySelectorAll("#new-status option").forEach(opt => {
          if (isAdmin || opt.value == "entnahme") {
            opt.disabled = false;
            opt.textContent.replace("🔒 ", "");
            opt.textContent.replace(" (nur für Admins)", "");

          } else {
            // Nur Entnahme ist erlaubt
            opt.disabled = true;
            opt.textContent  = "🔒 " + opt.textContent + " (nur für Admins)";
          }
        });
      }

      applyUserRights();

      document.getElementById('clear-selection-btn').addEventListener('click', () => {
        if (!preselectedSet.size) return;

        const confirmed = confirm("Möchten Sie die aktuelle Auswahl wirklich löschen?");
        if (!confirmed) return;

        preselectedSet.clear();
        document.querySelectorAll('.row-checkbox:checked').forEach(cb => cb.checked = false);
        updateFloatingBar();
      });

      // Dropdown-Logik für Logout
      const userMenuBtn = document.getElementById('userMenuBtn');
      const userDropdown = document.getElementById('userDropdown');
      if(userMenuBtn && userDropdown){
        userMenuBtn.addEventListener('click', (e) => {
          e.stopPropagation();
          const isOpen = userDropdown.classList.contains('opacity-100');
          userDropdown.classList.toggle('opacity-100', !isOpen);
          userDropdown.classList.toggle('pointer-events-auto', !isOpen);
          userDropdown.classList.toggle('scale-100', !isOpen);
          userDropdown.classList.toggle('opacity-0', isOpen);
          userDropdown.classList.toggle('pointer-events-none', isOpen);
          userDropdown.classList.toggle('scale-95', isOpen);
        });
        window.addEventListener('click', () => {
          userDropdown.classList.remove('opacity-100', 'pointer-events-auto', 'scale-100');
          userDropdown.classList.add('opacity-0', 'pointer-events-none', 'scale-95');
        });
        userDropdown.addEventListener('click', e => e.stopPropagation());
      }

      // Logout-Button aus Dropdown
      const logoutDropdownBtn = document.getElementById("logoutDropdownBtn");
      if (logoutDropdownBtn) {
        logoutDropdownBtn.addEventListener("click", () => {
          document.cookie = "jwt=; path=/; expires=Thu, 01 Jan 1970 00:00:00 GMT";
          window.location.href = './login_page?redirect=artinventarausgabe_page';
        });
      }

    // Toast-Komponente
    function showToast(message, type = 'info') {
      const id = 'toast_' + Date.now() + '_' + Math.floor(Math.random() * 1000);
      const container = document.getElementById('toast-container');

      const styleMap = {
        error: {
          bg: 'bg-red-50', border: 'border-red-200', text: 'text-red-800', icon: 'text-red-400',
          iconPath: 'M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z'
        },
        success: {
          bg: 'bg-green-50', border: 'border-green-200', text: 'text-green-800', icon: 'text-green-400',
          iconPath: 'M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z'
        },
        warning: {
          bg: 'bg-yellow-50', border: 'border-yellow-200', text: 'text-yellow-800', icon: 'text-yellow-400',
          iconPath: 'M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L3.732 16c-.77.833.192 2.5 1.732 2.5z'
        },
        info: {
          bg: 'bg-blue-50', border: 'border-blue-200', text: 'text-blue-800', icon: 'text-blue-400',
          iconPath: 'M13 16h-1v-4h-1m1-4h.01M12 2a10 10 0 100 20 10 10 0 000-20z'
        }
      };

      const style = styleMap[type] || styleMap.info;

      const toast = document.createElement('div');
      toast.id = id;
      toast.className = `flex items-start w-full p-4 border-l-4 rounded-md shadow ${style.bg} ${style.border} ${style.text} opacity-0 transition-opacity duration-300`;
      toast.innerHTML = `
        <svg class="w-6 h-6 mr-2 ${style.icon}" fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="${style.iconPath}" />
        </svg>
        <div class="flex-1 text-sm">${message}</div>
        <button onclick="removeToast('${id}')" class="ml-3 text-gray-400 hover:text-gray-700" aria-label="Schließen">
          <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" />
          </svg>
        </button>
      `;

      container.appendChild(toast);
      setTimeout(() => {
        toast.classList.add('opacity-100');
      }, 10);

      // Automatisch entfernen
      const timeout = type === 'error' ? 10000 : 5000;
      setTimeout(() => removeToast(id), timeout);
    }

    function removeToast(id) {
      const toast = document.getElementById(id);
      if (toast) {
        toast.classList.remove('opacity-100');
        toast.classList.add('opacity-0');
        setTimeout(() => toast.remove(), 300);
      }
    }

    window.showToast = showToast;
    window.removeToast = removeToast;

    </script>
  </body>
  </html>$html$ );
$BODY$ LANGUAGE sql;


-- Funktion zum Setzen des Status von Betriebsmitteln
-- Diese Funktion wird per PostgREST-API aufgerufen, um den Status von Betriebsmitteln zu ändern.
-- Sie erwartet eine Liste von Betriebsmittel-IDs, den neuen Status und optionale Parameter
CREATE OR REPLACE FUNCTION api.artinventar_ausgabe__status_set(
    artinventar_ids  integer[],
    status           varchar,
    params           jsonb DEFAULT null,
    _minr            integer DEFAULT null
  )
  RETURNS jsonb
  AS $$
  DECLARE
    _jwt_token varchar;
  BEGIN

    -- jwt aus Cookie validieren
    IF _minr IS NULL AND api_basic_auth.jwt_validate_from_cookie() THEN
      _minr = current_setting( 'request.jwt.claims', true )::jsonb ->> 'll_minr';
    ELSE
      RAISE EXCEPTION 'Es wurde keine Mitarbeiternummer angegeben.';
    END IF;

    -- Status setzen
    RAISE NOTICE 'Prüfung: %', status = 'entnahme';

    IF status = 'entnahme' THEN
      RAISE NOTICE 'entnahme: %', artinventar_ids;

      INSERT INTO artinventarausgabe (ainva_ainv_id, ainva_ausdat, ainva_ausan, ainva_ausort, ainva_ab_ix)
             SELECT artinventar_id, current_date, _minr, params ->> 'pr_ort', (params ->> 'pr_abix')::int
               FROM unnest(artinventar_ids) AS artinventar_id;

      UPDATE artinventar
         SET ainv_ort = coalesce(params ->> 'pr_ort', ''),
             ainv_abix =(params ->> 'pr_abix')::int,
             ainv_minr = _minr,
             ainv_auda = current_date,
             ainv_ibdat = coalesce(ainv_ibdat, current_date)
       WHERE ainv_id = ANY (artinventar_ids);

    ELSIF status = 'rueckgabe' THEN
      UPDATE artinventarausgabe
         SET ainva_rueckdat = current_date
       WHERE ainva_ainv_id = ANY (artinventar_ids);

      UPDATE artinventar
         SET ainv_ort = null,
             ainv_abix = null,
             ainv_minr = null,
             ainv_auda = null
       WHERE ainv_id = ANY (artinventar_ids);

    ELSIF status = 'zur kalibrierung' THEN
      UPDATE artinventar
         SET ainv_befu = 'Zur Kalibrierung',
             ainv_auda = current_date,
             ainv_ibdat = NULL,
             ainv_lpdat = NULL,
             ainv_npda = NULL,
             ainv_minr = NULL,
             ainv_ort = NULL,
             ainv_abix = NULL
       WHERE ainv_id = ANY (artinventar_ids);

    ELSIF status = 'von kalibrierung' THEN
      UPDATE artinventar
         SET ainv_befu = 'Unbenutzt',
             ainv_auda = null,
             ainv_lpdat = current_date
       WHERE ainv_id = ANY (artinventar_ids);

    END IF;

    RETURN jsonb_build_object('result', status);

  END $$ LANGUAGE plpgsql;
